# -*- coding: iso-8859-1 -*-
#*===========================================================================*#
#*                Copyright (c) 1997-2015 by miControl. All rights reserved. *#
#* File     : mc_interface_modbus.py                                         *#
#*                                                                           *#
#* Project  : Modbus                                                         *#
#* System   : Python                                                         *#
#* Version  : (see Version)                                                  *#
#* Company  : miControl                                                      *#
#*                                                                           *#
#* Author   : Kamil Kaczmarek (KK)                                           *#
#* Date     : 30.08.2010                                                     *#
#* Description: Example for communication with miControl devices via         *#
#*              Modbus RTU                                                   *#
#*                                                                           *#
#*===========================================================================*#
#* History  : 30.08.2010 KK: +Created                                        *#
#*                                                                           *#
#*===========================================================================*#
import time, struct, serial
from mc.dsa import *

Version = (1,0x00,0x00,0x00)

TableCRCHi = [0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
              0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,
              0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01,
              0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41,
              0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81,
              0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0,
              0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01,
              0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
              0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
              0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,
              0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01,
              0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
              0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
              0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,
              0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01,
              0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
              0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
              0x40]

TableCRCLo = [0x00, 0xC0, 0xC1, 0x01, 0xC3, 0x03, 0x02, 0xC2, 0xC6, 0x06, 0x07, 0xC7, 0x05, 0xC5, 0xC4,
              0x04, 0xCC, 0x0C, 0x0D, 0xCD, 0x0F, 0xCF, 0xCE, 0x0E, 0x0A, 0xCA, 0xCB, 0x0B, 0xC9, 0x09,
              0x08, 0xC8, 0xD8, 0x18, 0x19, 0xD9, 0x1B, 0xDB, 0xDA, 0x1A, 0x1E, 0xDE, 0xDF, 0x1F, 0xDD,
              0x1D, 0x1C, 0xDC, 0x14, 0xD4, 0xD5, 0x15, 0xD7, 0x17, 0x16, 0xD6, 0xD2, 0x12, 0x13, 0xD3,
              0x11, 0xD1, 0xD0, 0x10, 0xF0, 0x30, 0x31, 0xF1, 0x33, 0xF3, 0xF2, 0x32, 0x36, 0xF6, 0xF7,
              0x37, 0xF5, 0x35, 0x34, 0xF4, 0x3C, 0xFC, 0xFD, 0x3D, 0xFF, 0x3F, 0x3E, 0xFE, 0xFA, 0x3A,
              0x3B, 0xFB, 0x39, 0xF9, 0xF8, 0x38, 0x28, 0xE8, 0xE9, 0x29, 0xEB, 0x2B, 0x2A, 0xEA, 0xEE,
              0x2E, 0x2F, 0xEF, 0x2D, 0xED, 0xEC, 0x2C, 0xE4, 0x24, 0x25, 0xE5, 0x27, 0xE7, 0xE6, 0x26,
              0x22, 0xE2, 0xE3, 0x23, 0xE1, 0x21, 0x20, 0xE0, 0xA0, 0x60, 0x61, 0xA1, 0x63, 0xA3, 0xA2,
              0x62, 0x66, 0xA6, 0xA7, 0x67, 0xA5, 0x65, 0x64, 0xA4, 0x6C, 0xAC, 0xAD, 0x6D, 0xAF, 0x6F,
              0x6E, 0xAE, 0xAA, 0x6A, 0x6B, 0xAB, 0x69, 0xA9, 0xA8, 0x68, 0x78, 0xB8, 0xB9, 0x79, 0xBB,
              0x7B, 0x7A, 0xBA, 0xBE, 0x7E, 0x7F, 0xBF, 0x7D, 0xBD, 0xBC, 0x7C, 0xB4, 0x74, 0x75, 0xB5,
              0x77, 0xB7, 0xB6, 0x76, 0x72, 0xB2, 0xB3, 0x73, 0xB1, 0x71, 0x70, 0xB0, 0x50, 0x90, 0x91,
              0x51, 0x93, 0x53, 0x52, 0x92, 0x96, 0x56, 0x57, 0x97, 0x55, 0x95, 0x94, 0x54, 0x9C, 0x5C,
              0x5D, 0x9D, 0x5F, 0x9F, 0x9E, 0x5E, 0x5A, 0x9A, 0x9B, 0x5B, 0x99, 0x59, 0x58, 0x98, 0x88,
              0x48, 0x49, 0x89, 0x4B, 0x8B, 0x8A, 0x4A, 0x4E, 0x8E, 0x8F, 0x4F, 0x8D, 0x4D, 0x4C, 0x8C,
              0x44, 0x84, 0x85, 0x45, 0x87, 0x47, 0x46, 0x86, 0x82, 0x42, 0x43, 0x83, 0x41, 0x81, 0x80,
              0x40]

#==============================================================================
class ExcInterf (Exception):
   #---------------------------------------------------------------------------
   def __init__(self, err, description=""):
      self.Error       = err
      self.Description = description
   
   #---------------------------------------------------------------------------
   def __repr__(self):
      return "<ERROR(%d): %s>" % (self.Error, self.Description)
   
   #---------------------------------------------------------------------------
   def __str__(self):
      return "ERROR(%d): %s" % (self.Error, self.Description)

#==============================================================================
class Modbus(dsa):
   FUNCTION_READ_INPUT_REGISTER     = 0x04
   FUNCTION_WRITE_SINGLE_REGISTER   = 0x06
   FUNCTION_EXCEPTION_STATUS        = 0x07
   FUNCTION_DIAGNOSTIC              = 0x08
   FUNCTION_COMM_EVENT_COUNTER      = 0x0B
   FUNCTION_WRITE_MULTIPLE_REGISTER = 0x10
   FUNCTION_REPORT_SLAVE_ID         = 0x11
   FUNCTION_READ_FILE               = 0x14
   FUNCTION_WRITE_FILE              = 0x15
   
   #---------------------------------------------------------------------------
   def __init__(self,node_id,port=1,baudrate=9600,parity='N',stopbits=1,bytesize=8):
      '''
      BAUDRATES                               = (50,75,110,134,150,200,300,600,1200,1800,2400,4800,9600,19200,38400,57600,115200)
      PARITY_NONE, PARITY_EVEN, PARITY_ODD    = 'N', 'E', 'O'
      STOPBITS_ONE, STOPBITS_TWO              = (1, 2)
      FIVEBITS, SIXBITS, SEVENBITS, EIGHTBITS = (5,6,7,8)
      '''
      self.SerialCom = serial.Serial(port     = "COM%d" % port,
                                     baudrate = baudrate,
                                     parity   = parity,
                                     stopbits = stopbits,
                                     bytesize = bytesize)
      
      self.NodeId = node_id
      self.Debug  = False
   
   #---------------------------------------------------------------------------
   def __del__(self):
      if self.SerialCom.isOpen():
         self.SerialCom.close()
   
   #---------------------------------------------------------------------------
   def modbus_crc16 (self, s):
      crc_hi = 0xFF                 # high CRC byte initialized
      crc_lo = 0xFF                 # low CRC byte initialized
      
      for ch in s:
         index  = (crc_lo ^ ord(ch)) & 0xFF           # calculate the CRC
         crc_lo = (crc_hi ^ TableCRCHi[index]) & 0xFF
         crc_hi = TableCRCLo[index]
      
      return (crc_hi << 8) | crc_lo                   # return CRC
   
   #---------------------------------------------------------------------------
   def modbus_WriteFile(self, slave_nr, file_nr, record_nr, value, timeout):
      frame = struct.pack(">4B3H1I", slave_nr, self.FUNCTION_WRITE_FILE, 0x0B, 0x06, file_nr, record_nr, 0x0004/2, value)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "WriteFile     :",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_ReadFile(self, slave_nr, file_nr, record_nr, timeout=1000):
      frame = struct.pack(">4B3H", slave_nr, self.FUNCTION_READ_FILE, 0x07, 0x06, file_nr, record_nr, 0x0004/2)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "ReadFile      :",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_ReadInputRegister(self, slave_nr, address, quantity, timeout=1000):
      frame = struct.pack(">2B2H", slave_nr, self.FUNCTION_READ_INPUT_REGISTER, address, quantity)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "ReadInputReg  :",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_WriteSingleRegister(self, slave_nr, address, value, timeout=1000):
      frame = struct.pack(">2B2H", slave_nr, self.FUNCTION_WRITE_SINGLE_REGISTER, address, value)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "WriteSingleReg:",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_WriteMultipleRegisters(self, slave_nr, address, values=[], timeout=1000):
      
      quantity    = len(values)*2         # because the values are put into the var "values" as int32 -> 2 registers (2 x 2bytes) = 4 bytes (length of int32)
      byte_counts = quantity*2            # each register has 2 bytes
      
      frame = struct.pack(">2B2H1B", slave_nr, self.FUNCTION_WRITE_MULTIPLE_REGISTER, address, quantity, byte_counts)
      for value in values:
         frame += struct.pack(">2H", value>>16, value&0xFFFF)
         
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "WriteMultiRegs:",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_Diagnostic(self, slave_nr, subfunction, data, timeout=1000):
      frame = struct.pack(">2B2H", slave_nr, self.FUNCTION_DIAGNOSTIC, subfunction, data)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "Diagnostic   :",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_CommEventCounter(self, slave_nr, timeout=1000):
      frame = struct.pack(">2B", slave_nr, self.FUNCTION_COMM_EVENT_COUNTER)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "CommEventCnt :",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_ReportSlaveId(self, slave_nr, timeout=1000):
      frame = struct.pack(">2B", slave_nr, self.FUNCTION_REPORT_SLAVE_ID)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "ReportSlaveId :",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def modbus_ExceptionStatus(self, slave_nr, timeout=1000):
      frame = struct.pack(">2B", slave_nr, self.FUNCTION_EXCEPTION_STATUS)
      frame += struct.pack("<1H", self.modbus_crc16(frame))
      
      if self.Debug:
         print "ReportSlaveId :",
         for i in frame:
            print "0x%02X" % (ord(i)),
         print
      
      return self.SendCmd(frame, timeout)
   
   #---------------------------------------------------------------------------
   def SendCmd(self, frame, timeout):
      self.SerialCom.write(frame)
      
      ts = time.time()
      while not self.SerialCom.inWaiting():
         if time.time()- ts > timeout/1000.0:
            self.SerialCom.close()
            raise ExcInterf(-1, "[Modbus] timeout at wait for answer")
            break
         time.sleep(0.01)
      
      time.sleep(0.2)
      input = ""
      while self.SerialCom.inWaiting() > 0:
         input += self.SerialCom.read(1)
      
      if self.Debug:
         print "Response      :",
         for i in input:
            print "0x%02X" % (ord(i)),
         print
      
      return input
   
   #---------------------------------------------------------------------------
   def SdoRd(self, index, subindex, timeout=1000):
      node_id = self.NodeId
      
      value = 0
      resp = self.modbus_ReadFile(node_id, index, subindex, timeout)
      
      if (ord(resp[0]) == node_id):
         if ord(resp[1]) == 0x14:
            try:
               value = struct.unpack(">1L", resp[5:9])[0]
               err   = 0
            except:
               err = -1
         else:
            err = ord(resp[2])
      else:
         err = -1
      
      if err:
         if err == 0x0B:    txt = "Index does not exist"
         elif err == 0x0C:  txt = "Subindex does not exist"
         elif err == 0x0D:  txt = "Parameter has not write permission"
         elif err == 0x0E:  txt = "Parameter has not read permission"
         elif err == 0x0F:  txt = "Parameter value out of range"
         elif err == 0x0A:  txt = "Unknown error"
         else:              txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
      
      return value
   
   #---------------------------------------------------------------------------
   def SdoWr(self, index, subindex, value, timeout=1000):
      node_id = self.NodeId
      
      resp = self.modbus_WriteFile(node_id, index, subindex, value, timeout)
      
      if (ord(resp[0]) == node_id):
         if ord(resp[1]) == 0x15:
            try:
               value = struct.unpack(">1L", resp[5:9])[0]
               err   = 0
            except:
               err = -1
         else:
            err = ord(resp[2])
      else:
         err = -1
      
      if err:
         if err == 0x0B:    txt = "Index does not exist"
         elif err == 0x0C:  txt = "Subindex does not exist"
         elif err == 0x0D:  txt = "Parameter has not write permission"
         elif err == 0x0E:  txt = "Parameter has not read permission"
         elif err == 0x0F:  txt = "Parameter value out of range"
         elif err == 0x0A:  txt = "Unknown error"
         else:              txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
      
      return err
   
   #---------------------------------------------------------------------------
   def ReadInputRegister(self, address, quantity, timeout=1000):
      err = -255
      
      resp = self.modbus_ReadInputRegister(self.NodeId, address, quantity, timeout)
      
      if (ord(resp[0]) == self.NodeId):
         if ord(resp[1]) == self.FUNCTION_READ_INPUT_REGISTER:
            try:
               value = []
               
               for i in xrange(0,ord(resp[2])):
                  value += struct.unpack(">1B", resp[3+i])
               err   = 0
            except:
               err = -1
         else:
            err = -ord(resp[2])
      
      if err:
         
         if err == -1:     txt = "Invalid function code"
         elif err == -2:   txt = "Invalid data address"
         elif err == -3:   txt = "Invalid data value"
         elif err == -4:   txt = "Invalid function operation"
         else:             txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
      
      return value
   
   #---------------------------------------------------------------------------
   def WriteSingleRegister(self, address, value, timeout=1000):
      err = -255
      
      resp = self.modbus_WriteSingleRegister(self.NodeId, address, value, timeout)
      
      if (ord(resp[0]) == self.NodeId):
         if ord(resp[1]) == self.FUNCTION_WRITE_SINGLE_REGISTER:
            try:
               value_read = struct.unpack(">1H", resp[4:6])[0]            
               err   = 0
            except:
               err = -4
         else:
            err = -ord(resp[2])
      
      if err:
         if err == -1:     txt = "Invalid function code"
         elif err == -2:   txt = "Invalid data address"
         elif err == -3:   txt = "Invalid data value"
         elif err == -4:   txt = "Invalid function operation"
         else:             txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
   
   #---------------------------------------------------------------------------
   def WriteMultipleRegisters(self, address, values, timeout=1000):
      err = -255
      
      resp = self.modbus_WriteMultipleRegisters(self.NodeId, address, values, timeout)
      
      if (ord(resp[0]) == self.NodeId):
         if ord(resp[1]) == self.FUNCTION_WRITE_MULTIPLE_REGISTER:
            err = 0
         else:
            err = -ord(resp[2])
      
      if err:
         if err == -1:     txt = "Invalid function code"
         elif err == -2:   txt = "Invalid data address"
         elif err == -3:   txt = "Invalid data value"
         elif err == -4:   txt = "Invalid function operation"
         else:             txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
   
   #---------------------------------------------------------------------------
   def return_diagnostic(self, subfunction, data=0x0000, timeout=1000):
      value = None
      err   = -255
      
      resp = self.modbus_Diagnostic(slave_nr=self.NodeId, subfunction=subfunction, data=data, timeout=timeout)
      
      if (ord(resp[0]) == self.NodeId):
         if ord(resp[1]) == self.FUNCTION_DIAGNOSTIC:
            try:
               value = struct.unpack(">1H", resp[4:6])[0]            
               err   = 0
            except:
               err = -1
         else:
            err = -ord(resp[2])
      
      if err:
         if err == -1:     txt = "Invalid function code"
         elif err == -2:   txt = "Invalid data address"
         elif err == -3:   txt = "Invalid data value"
         elif err == -4:   txt = "Invalid function operation"
         else:             txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
      
      return value
   
   #---------------------------------------------------------------------------
   def DiagReturnQueryData(self, data=0x0000, timeout=1000):
      return self.return_diagnostic(0x0000, data=data, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def DiagRestartCommunicationsOption(self, data=0x0000, timeout=1000):
      return self.return_diagnostic(0x0001, data=data, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def DiagReturnDiagnosticRegister(self, timeout=1000):
      return self.return_diagnostic(0x0002, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def DiagClearCountersAndDiagnosticRegister(self, timeout=1000):
      return self.return_diagnostic(0x000A, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def DiagReturnBusMessageCount(self, timeout=1000):
      return self.return_diagnostic(0x000B, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def DiagReturnBusCommunicationErrorCount(self, timeout=1000):
      return self.return_diagnostic(0x000C, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def DiagReturnBusExceptionErrorCount(self, timeout=1000):
      return self.return_diagnostic(0x000D, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def DiagReturnSlaveMessageCount(self, timeout=1000):
      return self.return_diagnostic(0x000E, timeout=timeout)
   
   #---------------------------------------------------------------------------
   def GetCommEventCounter(self, timeout=1000):
      status   = None
      evt_cnt  = None
      err      = -255
      
      resp = self.modbus_CommEventCounter(self.NodeId, timeout=timeout)
      
      if (ord(resp[0]) == self.NodeId):
         if ord(resp[1]) == self.FUNCTION_COMM_EVENT_COUNTER:
            try:
               status  = struct.unpack(">1H", resp[2:4])[0]
               evt_cnt = struct.unpack(">1H", resp[4:6])[0]
               err   = 0
            except:
               err = -1
         else:
            err = -ord(resp[2])
      
      if err:
         if err == -1:     txt = "Invalid function code"
         elif err == -2:   txt = "Invalid data address"
         elif err == -3:   txt = "Invalid data value"
         elif err == -4:   txt = "Invalid function operation"
         else:             txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
      
      return status, evt_cnt
   
   #---------------------------------------------------------------------------
   def ReportSlaveId(self, timeout=1000):
      slave_id    = None
      run_status  = None
      err         = -255
      
      resp = self.modbus_ReportSlaveId(self.NodeId, timeout=timeout)
      
      if (ord(resp[0]) == self.NodeId):
         if ord(resp[1]) == self.FUNCTION_REPORT_SLAVE_ID:
            try:
               slave_id   = struct.unpack(">1B", resp[3])[0]
               run_status = struct.unpack(">1B", resp[4])[0]
               err        = 0
            except:
               err = -1
         else:
            err = -ord(resp[2])
      
      if err:
         if err == -1:     txt = "Invalid function code"
         elif err == -2:   txt = "Invalid data address"
         elif err == -3:   txt = "Invalid data value"
         elif err == -4:   txt = "Invalid function operation"
         else:             txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
      
      return slave_id, run_status
   
   #---------------------------------------------------------------------------
   def ExceptionStatus(self, timeout=1000):
      status   = None
      err      = -255
      
      resp = self.modbus_ExceptionStatus(self.NodeId, timeout=timeout)
      
      if (ord(resp[0]) == self.NodeId):
         if ord(resp[1]) == self.FUNCTION_EXCEPTION_STATUS:
            try:
               status = struct.unpack(">1B", resp[2])[0]
               err        = 0
            except:
               err = -1
         
         else:
            err = -ord(resp[2])
      
      if err:
         if err == -1:     txt = "Invalid function code"
         elif err == -2:   txt = "Invalid data address"
         elif err == -3:   txt = "Invalid data value"
         elif err == -4:   txt = "Invalid function operation"
         else:             txt = "Bad response"
         
         raise ExcInterf(err, "InterfaceError: %s" % txt)
      
      return status

#==============================================================================
if __name__ == "__main__":   
   SlaveId = 3                                                 # Modbus slave address
   
   # Initialize Modbus slave
   mb = Modbus(node_id=SlaveId,port=1,baudrate=9600,parity='N',stopbits=1,bytesize=8)
   
   mb.Debug = True                                             # Debug = True => prints all data frames
   
   print "---------------"
    
   # Example: Modbus mapping
   if 0:
      mb.SdoWr(0x3C02,0x00,8)             # Amount of mapping entries
      mb.SdoWr(0x3C02,0x01,0x30001020)    # Obj: Index=3000h Subindex=0x10 Length=20h=32d
      mb.SdoWr(0x3C02,0x02,0x30001120)    # Obj: Index=3000h Subindex=0x11 Length=20h=32d
      mb.SdoWr(0x3C02,0x03,0x30001220)    # Obj: Index=3000h Subindex=0x13 Length=20h=32d
      mb.SdoWr(0x3C02,0x04,0x30000120)    # Obj: Index=3000h Subindex=0x01 Length=20h=32d
      mb.SdoWr(0x3C02,0x05,0x30007020)    # Obj: Index=3000h Subindex=0x70 Length=20h=32d
      mb.SdoWr(0x3C02,0x06,0x30010020)    # Obj: Index=3001h Subindex=0x00 Length=20h=32d
      mb.SdoWr(0x3C02,0x07,0x30020020)    # Obj: Index=3002h Subindex=0x00 Length=20h=32d
      mb.SdoWr(0x3C02,0x08,0x37620020)    # Obj: Index=3762h Subindex=0x00 Length=20h=32d
   
   # Writes a Holding register (Function code = 0x06)
   if 0:
      mb.WriteSingleRegister(1, 0x7654)                           # 3000.10h = 0x00007654
      
   # Writes in multiple Holding registers (Function code = 0x10)
   if 0:   
      mb.WriteMultipleRegisters(0, values=[0x01015679, 0x1239])   # 3000.10h = 0x01015679  3000.11h = 0x00001239
      
   # Reads multiple registers (Function code = 0x04)
   if 0:
      mb.ReadInputRegister(0,4)                                   # Reads 3000.10h and 3000.11h 
   
   # Diagnosis functions (Function code = 0x08)
   if 0:
      # Query request (sub function = 0x0000)
      value = mb.DiagReturnQueryData(0x1234)
      print "QueryData", hex(value)
      print
      
      # Read diagnosis register (sub function = 0x0002)
      value = mb.DiagReturnDiagnosticRegister()
      print "DiagnosticRegister:", value
      print
      
      if 0:
         # Clear counter and diagnosis register (sub function = 0x000A)
         mb.DiagClearCountersAndDiagnosticRegister()
         print "Clearing Counter and DiagnosticRegister!"
         print
      
      # Read bus message counter (sub function = 0x000B)
      value = mb.DiagReturnBusMessageCount()
      print "BusMessageCount:", value
      print
      
      # Read bus error counter for CRC errors (sub function = 0x000C)
      value = mb.DiagReturnBusCommunicationErrorCount()
      print "BusCommunicationErrorCount:", value
      print
      
      # Read bus error counter for exceptions (sub function = 0x000D)
      value = mb.DiagReturnBusExceptionErrorCount()
      print "BusExceptionErrorCount:", value
      print
      
      # Read message counter (sub function = 0x000E)
      value = mb.DiagReturnSlaveMessageCount()
      print "SlaveMessageCount:", value
      print
   
   # Read communication counter (Function code = 0x0B)
   if 0:
      status, evt_cnt = mb.GetCommEventCounter()
      print "Status", status, "CommEventCounter", evt_cnt
   
   # Read SlaveId (Function code = 0x11)
   if 0:
      slave_id, run_status = mb.ReportSlaveId()
      print "SlaveId", slave_id, "RunIndicatorStatus", hex(run_status)
   
   # Read error register (Function code = 0x07)
   if 0:
      value = mb.ExceptionStatus()
      print "ExceptionStatus", value
      print
   